# File Viewing Module for hex editor
package Hex::FileSelect;
use strict;
use Win32::Console;
use File::Spec;
use Time::HiRes qw(usleep);
use Cwd;
use Cwd qw(abs_path);

sub new {
    my $class = shift;
    my $self = {
        Title => 'Select File',  #TitleBar text
        Start => getcwd(),  #Start Dir
        CurDir => '', #Current Dir
        Dir => [],    #Dir Contents
        Offset => 0,  #Filelist display offset
        Highlighted => 0, #Highlighted file offset
        Selected => undef, #What the user actually chose
        DefaultStatus => 'Choose a file or press SPACEBAR to type it in manually.',
        Input => $Hex::INPUT_CONSOLE,
        Output => new Win32::Console(),
        Handler => {  #Key Code => Handler Sub
            13 => \&_kEnter,
            27 => \&_kEsc,
            32 => \&_kSpace,
            33 => \&_kPgUp,
            34 => \&_kPgDown,
            37 => \&_kLeft,
            38 => \&_kUp,
            39 => \&_kRight,
            40 => \&_kDown,
            116=> \&_kF5,
        },    
    };
    bless($self, $class);
    return $self;
}

# Returns a string representing the path to the chosen file.
sub Run {
    my $self = shift;
    $self->{Input}->Cursor(0,0,0,0);    
    $self->{CurDir} = '';
    $self->{Dir} = [];
    $self->{Offset} = 0;
    $self->{Highlighted} = 0;
    $self->{Selected} = undef;

    $self->{Output}->Size(80, 50);
    $self->{Output}->Window(1, 0, 0, 640, 480);
    $self->{Output}->Cursor(-1, -1, 100, 0);
    $self->{Input}->Mode(ENABLE_PROCESSED_INPUT);
    $self->{Output}->FillAttr($main::BG_BLACK | $main::FG_LIGHTCYAN, 80 * 50, 0, 0);

   #Title Bar
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_WHITE, 80, 0, 0);
    $self->{Output}->FillChar(' ', 80, 0, 0);
    $self->{Output}->WriteChar($self->{Title}, 0, 0);
   #Status Bar
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_WHITE, 80, 0, 49);
   #Swap in display buffer
    $self->{Output}->Display();
   #Read Start Directory
    $self->{CurDir} = $self->{Start};

    $self->_DisplayCurDir();
    $self->_LoadCurDir();
    $self->_DisplayList();
    $self->_Highlight();
    $self->_StatusBar($self->{DefaultStatus});
    return $self->_MainLoop();
}

#Returns a path/filname or '' on cancel
sub _MainLoop() {
    my($self) = @_;
    my $events;
    my @event;
    $self->{Input}->Flush();
    #0 Event Type (1 = keyboard)
    #1 Key Down (TRUE = Pressed, FALSE = Released)
    #2 Repeat
    #3 Virtual Key
    #4 Virtual Scan
    #5 ASCII Char (0 if non-character)
    #6 Control Key State
    while(1) {
        if($self->{Input}->GetEvents()) {
            @event = $self->{Input}->Input();
#                $self->_StatusBar("KC: $event[3]");
#                sleep(1);
            if($event[1] && defined $self->{Handler}->{$event[3]}) {
                $self->{Handler}->{$event[3]}->($self);
                if(defined $self->{Selected}) {
                    return $self->{Selected};
                }    
            }    
        }
        1;
    }    
}

sub Title {
    my($self, $title) = @_;
    if(defined $title) {
       #Set Title
        $self->{Title} = $title;
        return;
    }else{
       #Get Title
        return $self->{Title};
    }
}

sub Start {
    my($self, $start) = @_;
    if(defined $start) {
       #Set Start
        $self->{Start} = $start;
        return;
    }else{
       #Get Start
        return $self->{Start};
    }
}

###############
## Key Handlers
###############

sub _kEnter {
    my($self) = @_;
    my $sel = $self->{Dir}->[$self->{Highlighted}];
    my $path = File::Spec->catfile($self->{CurDir}, $sel->[0]);
    if($sel->[1]) {   #dir
        $self->{CurDir} = abs_path($path);
        $self->{Offset} = 0;
        $self->{Highlighted} = 0;
        $self->_DisplayCurDir();
        $self->_LoadCurDir();
        $self->_DisplayList();
        $self->_Highlight();
    }else{
        $self->{Selected} = $path;
    }    
}

sub _kEsc {
    my($self) = @_;
    $self->{Selected} = '';
}

sub _kSpace {
    my($self) = @_;
    my @event;
    my $done;
    my $newpath;
    $self->_StatusBar("New File/Directory: ");
    while(1) {
        while($self->{Input}->GetEvents()) {
            @event = $self->{Input}->Input();
            if($event[1] && $event[3] == 8){
                chop $newpath;
            }elsif($event[1] && $event[3] == 13){
                $done = 1;
                last;
            }elsif($event[1] && $event[3] == 27){
                $done = 1;
                $newpath = '';
                last;
            }elsif($event[1] && $event[5]) {
                $newpath .= chr($event[5]);
            }
            
            if(length($newpath) > 60) {
                $self->_StatusBar(
                 "New File/Directory: ..." .
                 substr($newpath, length($newpath) - 57));
            }else{
                $self->_StatusBar("New File/Directory: $newpath");
            }
        }
        last if($done);
        1;
    }
    
    #done entering new path
    if($newpath ne '') {
        if(-d $newpath) {
            $self->{CurDir} = $newpath;
            $self->{Offset} = 0;
            $self->{Highlighted} = 0;
            $self->_DisplayCurDir();
            $self->_LoadCurDir();
            $self->_DisplayList();
            $self->_Highlight();
        }elsif(-e $newpath){
            $self->{Selected} = $newpath;
        }else{
            $self->_StatusBar("Invalid Path");
            usleep(500000);
        }    
    }
    $self->_StatusBar($self->{DefaultStatus});
}

sub _kPgUp {
    my($self) = @_;
    if($self->{Highlighted} == 0) {
        #DO SQUAT
    }else{
        $self->{Highlighted} -= 47;
        $self->{Highlighted} = 0
         if($self->{Highlighted} < 0);
        if($self->{Highlighted} < $self->{Offset}) {
            $self->{Offset} = $self->{Highlighted};
            $self->_DisplayList();
        }
        $self->_Highlight();
    }

}

sub _kPgDown {
    my($self) = @_;
    if($self->{Highlighted} == $#{ $self->{Dir} }) {
        #DO SQUAT
    }else{
        $self->{Highlighted} += 47;
        $self->{Highlighted} = $#{ $self->{Dir} }
         if($self->{Highlighted} > $#{ $self->{Dir} });
        if($self->{Highlighted} > $self->{Offset} + 46) {
            $self->{Offset} = $self->{Highlighted} - 46;
            $self->_DisplayList();
        }
        $self->_Highlight();
    }    
    
}

sub _kLeft {
    my($self) = @_;
#    my($vol, $dirs, undef) = File::Spec->splitpath($self->{CurDir}, 1);
#    my @dirs = File::Spec->splitdir($dirs);
#    pop @dirs;
    $self->{CurDir} = abs_path(File::Spec->catfile($self->{CurDir}, ".."));
    $self->{Offset} = 0;
    $self->{Highlighted} = 0;
    $self->_DisplayCurDir();
    $self->_LoadCurDir();
    $self->_DisplayList();
    $self->_Highlight();
}

sub _kUp {
    my($self) = @_;
    if($self->{Highlighted} == 0) {
        #DO SQUAT
    }else{
        $self->{Highlighted}--;
        if($self->{Highlighted} < $self->{Offset}) {
            $self->{Offset} = $self->{Highlighted};
            $self->_DisplayList();
        }
        $self->_Highlight();
    }     
}

sub _kRight {
    my($self) = @_;
    $self->_kEnter();
}

sub _kDown {      #list 47 entries long
    my($self) = @_;
    if($self->{Highlighted} == $#{ $self->{Dir} }) {
        #DO SQUAT
    }else{
        $self->{Highlighted}++;
        if($self->{Highlighted} > $self->{Offset} + 46) {
            $self->{Offset} = $self->{Highlighted} - 46;
            $self->_DisplayList();
        }
        $self->_Highlight();
    }    
}

sub _kF5 {
    my($self) = @_;
    $self->{Offset} = 0;
    $self->{Highlighted} = 0;
    $self->_LoadCurDir();
    $self->_DisplayList();
    $self->_Highlight;
}

sub _DisplayCurDir {
    my($self) = @_;
    $self->{Output}->FillChar(' ', 80, 0, 1);
    if(length $self->{CurDir} > 80) {
        $self->{Output}->WriteChar(
         "..." . substr($self->{CurDir}, length($self->{CurDir}) - 77), 0, 1)
    }else{
        $self->{Output}->WriteChar($self->{CurDir}, 0, 1)        
    }
}

sub _DisplayList {
    my($self) = @_;
    my $line = 2;
   #Range to display 
    my $first = $self->{Offset};
    my $last;
    if($first + 46 > @{ $self->{Dir} }) {
        $last = $#{ $self->{Dir} }
    }else{
        $last = $first + 46;
    }
    $self->{Output}->FillChar(' ', 80 * 47, 0, 2);
    foreach( @{ $self->{Dir} }[$first .. $last] ) {
        if($_->[1]) {
            $self->{Output}->WriteChar("-> $_->[0]", 0, $line);
        }else{
            $self->{Output}->WriteChar($_->[0], 3, $line);
        }
        $line++;
        last if($line > 49);
    }
}

sub _LoadCurDir {
    my($self) = @_;
    @{ $self->{Dir} } = ();
    opendir(DIR, $self->{CurDir}) || die "Unable to open $self->{Start}: $!\n";
    my @dir = readdir(DIR);
    my $filenum = 0;
    closedir(DIR);

    foreach(@dir) {
        if(-d File::Spec->catfile($self->{CurDir}, $_)) {
            $self->{Dir}->[$filenum] = [$_, 1];
        }else{
            $self->{Dir}->[$filenum] = [$_, 0];
        }
        $filenum++;
    }
   #Sort alphabetically
    @{ $self->{Dir} } = sort { $a->[0] cmp $b->[0] } @{ $self->{Dir} };
   #Put Dir's first
    my(@dirs, @files);
    foreach(@{ $self->{Dir} }) {
        if($_->[1]) {
            push(@dirs, $_);
        }else{
            push(@files, $_);
        }
    }
    @{ $self->{Dir} } = (@dirs, @files);
}

sub _Highlight {
    my($self) = @_;
    $self->{Output}->FillAttr($main::BG_BLACK | $main::FG_CYAN, 80 * 47, 0, 2);
    $self->{Output}->FillAttr($main::BG_LIGHTRED | $main::FG_WHITE, 80, 0, 
     2 + $self->{Highlighted} - $self->{Offset});
}

sub _StatusBar {
    my($self, $text) = @_;
    $self->{Output}->FillChar(' ', 80, 0, 49);
    $self->{Output}->WriteChar($text, 0, 49);
}

1;
