#!/usr/bin/perl
package Hex::Editor;
use strict;
use FindBin;
use Hex::FileSelect;
use Hex::TableFile;
use Hex::MessageBox;
use Hex::Goto;
use Hex::Replace;
use Hex::Find;
use Hex::RelSearch;
use Win32::Console;
use Math::BaseCalc;
use File::Spec;
use File::Copy;

our $SIG_INT = 0;
our $SIG_QUIT = 0;
$SIG{INT} = sub {$SIG_INT = 1};
$SIG{QUIT}= sub {$SIG_QUIT= 1};

sub new {
    my $class = shift;
    my $self = {
        Input => $Hex::INPUT_CONSOLE,
        Output => new Win32::Console(),
        Table => new Hex::TableFile,
        Base16 => new Math::BaseCalc(digits => 'HEX'),
        Base2  => new Math::BaseCalc(digits => 'bin'),
        Editing =>  0, #bool value true, if waiting for second spacebar press
        Busy    =>  0, #flag used to control "(Busy)" in corner
        Modified => 0,
        LastFind => '', #data returned from last Ctrl-F
        LastRelSearch => [],#data returned from last Ctrl-R
        Favorites => [], #array of favorites
        Filename => '',
        Path =>     '',
        Filesize => 0,
        TblFilename => '',
        TblPath     => '',
        Data =>     '',
        Offset =>   0,
        BPP => 16 * 46, #bytes per page
        Cursor   =>  0, #for navigation
        CursorAttrBuf => '', #chars representing attributes under highlighted bytes
        LastCursor => 0,
        SelStart =>  0, #for editing
        SelLength => 1,
        Handler => {   # KK => Subref
            '13' => \&_kEnter,
            '27' => \&_kEsc,
            '32' => \&_kSpace,  #edit
            '33' => \&_kPgUp,
            '34' => \&_kPgDown,
            '35' => \&_kEnd,
            '36' => \&_kHome,
            '37' => \&_kLeft,
            '38' => \&_kUp,
            '39' => \&_kRight,
            '40' => \&_kDown,
            '48' => sub {$_[0]->_GotoFavorite(0)},
            '49' => sub {$_[0]->_GotoFavorite(1)},
            '50' => sub {$_[0]->_GotoFavorite(2)},
            '51' => sub {$_[0]->_GotoFavorite(3)},
            '52' => sub {$_[0]->_GotoFavorite(4)},
            '53' => sub {$_[0]->_GotoFavorite(5)},
            '54' => sub {$_[0]->_GotoFavorite(6)},
            '55' => sub {$_[0]->_GotoFavorite(7)},
            '56' => sub {$_[0]->_GotoFavorite(8)},
            '57' => sub {$_[0]->_GotoFavorite(9)},
            '114'=> \&_kF3,     #repeat last find
            '115'=> \&_kF4,     #repeat last relative search
            '^35'=> \&_kCtrlEnd,
            '^36'=> \&_kCtrlHome,
            '^48' => sub {$_[0]->_StoreFavorite(0)},
            '^49' => sub {$_[0]->_StoreFavorite(1)},
            '^50' => sub {$_[0]->_StoreFavorite(2)},
            '^51' => sub {$_[0]->_StoreFavorite(3)},
            '^52' => sub {$_[0]->_StoreFavorite(4)},
            '^53' => sub {$_[0]->_StoreFavorite(5)},
            '^54' => sub {$_[0]->_StoreFavorite(6)},
            '^55' => sub {$_[0]->_StoreFavorite(7)},
            '^56' => sub {$_[0]->_StoreFavorite(8)},
            '^57' => sub {$_[0]->_StoreFavorite(9)},
            '^70'=> \&_kCtrlF, #Find
            '^71'=> \&_kCtrlG, #Goto
            '^79'=> \&_kCtrlO, #Open
            '^81'=> \&_kCtrlQ, #Quit
            '^82'=> \&_kCtrlR, #Relative Searcher
            '^83'=> \&_kCtrlS, #Save
            '^84'=> \&_kCtrlT, #Select Table File
        },
        Colors => { #colors for different length table file entries
            0 => $main::FG_GRAY | $main::BG_BLACK,
            1 => $main::FG_LIGHTRED   | $main::BG_BLACK,
            2 => $main::FG_LIGHTBLUE  | $main::BG_BLACK,
            3 => $main::FG_LIGHTGREEN | $main::BG_BLACK,
            4 => $main::FG_LIGHTMAGENTA | $main::BG_BLACK,
            5 => $main::FG_LIGHTCYAN  | $main::BG_BLACK,
            6 => $main::FG_YELLOW | $main::BG_BLACK,
        },
    };
    bless($self, $class);
    return $self;
}

sub Run {
    my($self, $path) = @_;
   #Initialization
    $self->{Path} = $path;
    $self->{Offset} = 0;
    $self->{SelStart} = 0;
    $self->{SelLength} = 1;

    $self->_Load($self->{Path});
    $self->_Draw();
    $self->_LoadFirstTableFile();
    $self->_LoadFavorites();
    $self->_Stats();
    $self->_Refresh();
    $self->_Highlight();
    $self->{Input}->Flush();

#### MAIN LOOP ####
#0 Event Type (1 = keyboard)
#1 Key Down (TRUE = Pressed, FALSE = Released)
#2 Repeat
#3 Virtual Key
#4 Virtual Scan
#5 ASCII Char (0 if non-character)
#6 Control Key State 40 = down, 38 is up i think but who cares

    while(1) {
       #Signals
        if($SIG_INT) {
            my $mbox = new Hex::MessageBox(
                           "Caught interrupt signal",
                           "Are you sure you want to end the program?",
                           qw(Yes No));
            if($mbox->Run() == 0) {
                $self->_EndProg();
            }else{
                #Do nothing
                $SIG_INT = 0;                
            }
        }

        if($SIG_QUIT) {
            my $mbox = new Hex::MessageBox(
                           "Caught quit signal",
                           "Are you sure you want to end the program?",
                           qw(Yes No));
            if($mbox->Run() == 0) {
                $self->_EndProg();
            }else{
                #Do nothing
                $SIG_QUIT = 0;
            }
        }    

       #Key Presses
        if($self->{Input}->GetEvents()) {
            my @event = $self->{Input}->Input();

           #Test for CTRL-KEY's  
            if($event[1]
              && ($event[6] & LEFT_CTRL_PRESSED || $event[6] & RIGHT_CTRL_PRESSED)
              && defined $self->{Handler}->{"^$event[3]"}) {
                $self->{Handler}->{"^$event[3]"}->($self);
            }
           #Text for Regular Keys
            elsif($event[1] && defined $self->{Handler}->{$event[3]}) {
                $self->{Handler}->{$event[3]}->($self);
            }
        }
        1;
    }
}

#### HANDLERS #

sub _kEnter {
    my($self) = @_;
}
sub _kEsc {
    #No Function
}
sub _kSpace {
    #Replacement
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();
    if($self->{Editing}) {
        my $replace = new Hex::Replace();
        my $replacement = $replace->Run($self->{Table}, $self->{SelLength});
        if($self->{Cursor} < $self->{SelStart}) {
            $self->{SelStart} = $self->{SelStart} - ($self->{SelLength} - 1);
        }
        if(length($replacement) > $self->{SelLength}) {
            my $msg = new Hex::MessageBox(
            "Replacement Error",
            "The replacement you specified cannot fit into the selection you\n".
            "have chosen. Either choose a larger selection, use fewer bytes,\n".
            "or h4xx0r your way to another solution.",
            qw(OK));
            $msg->Run();
        }
        elsif(length($replacement) < $self->{SelLength}) {
            my $msg = new Hex::MessageBox(
            "Warning!",
            "The replacement you specified is smaller than the size of your\n".
            "selection. Do you wish to only replace as much data as the\n".
            "replacement is long?",
            qw(No Yes));
            if($msg->Run() == 1) {
                $self->_Replace($replacement);
            }
        }
        elsif($replacement ne '') {
            $self->_Replace($replacement);
        }

        $self->{Editing} = 0;
        $self->{SelLength} = 1;
    }else{
        $self->{Editing} = 1;
        $self->{SelStart} = $self->{Cursor};
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _Replace {
    my($self, $replacement) = @_;
    substr($self->{Data}, $self->{SelStart}, length($replacement), $replacement);
    $self->{Modified} = 1;
    $self->_Refresh();
    $self->_Highlight();
}
sub _kPgUp {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();
    if($self->{Offset} > 0) {
        if($self->{Offset} - $self->{BPP} < 0) { #goto 0
            $self->{Offset} = 0;
        }else{ #go a whole page
            $self->{Offset} -= $self->{BPP};
        }
        $self->{Cursor} = $self->{Offset};
        $self->_Refresh();
        $self->_Highlight();
    }else{
        #we're at the top
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kPgDown {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();
    if($self->{Offset} + $self->{BPP} <= $self->{Filesize} - 1) {
        if($self->{Offset} + 2 * $self->{BPP} > $self->{Filesize}) { #special adjust
            $self->{Offset} = $self->_GetLastOffset();
        }else{ #room for full page
            $self->{Offset} += $self->{BPP};
        }
        $self->{Cursor} = $self->{Offset};
        $self->_Refresh();
        $self->_Highlight();
    }else{
        #Do nothing.
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kHome {
    my($self) = @_;
    $self->{Cursor} = $self->{Cursor} & (~0^0b1111);
    $self->_Highlight();
    $self->_Stats();
}
sub _kEnd {
    my($self) = @_;
    my $newloc = $self->{Cursor} | 0b1111;
    $newloc = $self->{Filesize} - 1 if($newloc >= $self->{Filesize});
    $self->{Cursor} = $newloc;
    $self->_Highlight();
    $self->_Stats();
}
sub _kCtrlHome {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    $self->{Cursor} = 0;
    if($self->{Offset} > 0) {
        $self->{Offset} = 0;
        $self->_Refresh();
    }else{
    }
    $self->_Highlight();
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kCtrlEnd {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    $self->{Cursor} = $self->{Filesize} - 1;
    my $lastoffset = $self->_GetLastOffset();
    if($self->{Offset} < $lastoffset) {
        $self->{Offset} = $lastoffset;
        $self->_Refresh();
    }else{
    }

    $self->_Highlight();
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kLeft {
    my($self) = @_;
    if($self->{Cursor} > 0) {
        if($self->{Cursor} == $self->{Offset}) {
            $self->_ScrollUp();
        }else{
        }
        $self->{Cursor}--;
        $self->_Highlight();
        $self->_Stats();
    }else{
        #Do..... nothing
    }
}
sub _kRight {
    my($self) = @_;
    if($self->{Cursor} < $self->{Filesize} - 1) {
        if($self->{Cursor} == $self->{Offset} + $self->{BPP} - 1) {
            $self->_ScrollDown();
        }else{
        }
        $self->{Cursor}++;
        $self->_Highlight();
        $self->_Stats();
    }else{
        #Do..... nothing
    }
}
sub _kUp {
    my($self) = @_;
    if($self->{Cursor} >= 16) {
        if($self->{Cursor} < $self->{Offset} + 16) {
            $self->_ScrollUp();
        }else{
        }
        $self->{Cursor} -= 16;
        $self->_Highlight();
        $self->_Stats();
    }else{
        #Do..... nothing
    }
}
sub _kDown {
    my($self) = @_;
    my $oddbytes = $self->{Filesize} % 16;
    my $xtraline = 0;
    $xtraline = 16 unless($self->{Filesize} % 16);

    if($self->{Cursor} < $self->{Filesize} - $oddbytes - $xtraline) {
        if($self->{Cursor} > $self->{Offset} + $self->{BPP} - 17) {
            $self->_ScrollDown();
        }else{
        }

        if($self->{Cursor} + 16 > $self->{Filesize} - 1) {
            $self->{Cursor} = $self->{Filesize} - 1;
        }else{
            $self->{Cursor} += 16;
        }
        $self->_Highlight();
        $self->_Stats();
    }else{
        #Do..... nothing
    }
}
sub _kCtrlF { # Find
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    my $find = new Hex::Find();
    my $query = $find->Run($self->{Table});
    $self->{LastFind} = $query;
    my $is_at = $self->_FindNext($self->{Cursor}, $query);
    if(defined $is_at) {
        $self->_Goto($is_at);
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kF3 {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    my $is_at = $self->_FindNext($self->{Cursor} + 1, $self->{LastFind});
    if(defined $is_at) {
        $self->_Goto($is_at);
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _FindNext {
    my($self, $from, $query) = @_;
    my $lookin = substr($self->{Data}, $from);
    $lookin =~ /\Q$query\E/g;
    return undef unless(defined pos($lookin));
    return pos($lookin) - length($query) + $from;
}
sub _kCtrlG {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    my $goto = new Hex::Goto();
    my($offset, $from) = $goto->Run();
    my %from = (0=>'Beginning', 1=>'Current Position', 2=>'End');
    if(defined $offset) { #undef = cancel
        if($from == 1) {  # 0 case not needed cuz then offset is fine
            $offset = $self->{Cursor} + $offset;
        }
        elsif($from == 2) {
            $offset = $self->{Filesize} - 1 - $offset;
        }
        if($offset < $self->{Filesize} && $offset >= 0) {
            $self->_Goto($offset);
        }else{
            my $mbox = new Hex::MessageBox(
               "Invalid Location",
               "The offset you specified ($offset) does not exist!",
               qw(OK));
            $mbox->Run();
        }
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kCtrlO {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    $self->_AskToSave();

    my $fs = new Hex::FileSelect();
    $fs->Start($Hex::ROM_DIR);
    $fs->Title("Select new file to open.");
    my $path = $fs->Run();

    if($path ne '') {
        $self->_Load($path);
        $self->{Offset} = 0;
        $self->{Cursor} = 0;
        $self->{SelStart} = 0;
        $self->{SelLength} = 1;
        $self->_Stats();
        $self->_Refresh();
        $self->_Highlight();
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kCtrlQ {
    my($self) = @_;
    my $mbox = new Hex::MessageBox(
                   "Ctrl-Q Pressed!",
                   "Are you sure you want to quit?",
                   qw(Yes No));
    if($mbox->Run() == 0) {
        $self->_AskToSave();
        $self->_EndProg();
    }else{
        #Do nothing
    }
}
sub _kCtrlR { #relative search
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    my $rel = new Hex::RelSearch();
    my @values = $rel->Run($self->{Table});
    @{ $self->{LastRelSearch} } = @values;
    my $is_at = $self->_FindNextRel($self->{Cursor}, @values);

    if(defined $is_at) {
        $self->_Goto($is_at);
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kF4 {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();
    my $is_at = undef;
    if(@{ $self->{LastRelSearch} }) {
        $is_at = $self->_FindNextRel($self->{Cursor} + 1, @{ $self->{LastRelSearch} });
    }
    if(defined $is_at) {
        $self->_Goto($is_at);
    }
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _FindNextRel {              #warnings can easily be thrown here, but
                                #it's ok in this case. Undef and '' values
                                #won't mess up behavior.
    my($self, $from, @values) = @_;
    my $lookin = substr($self->{Data}, $from);
    my $startat = 0;
    my $matched = 1;
    while($startat <= length($lookin) - (@values + 1)) {
        my $curbyte = 0;
        foreach(0..$#values) {
            my $bytepair = substr($lookin, $startat + $curbyte, 2);
            my($val1, $val2) = unpack('C*', $bytepair);
            if($val2 == $val1 + $values[$_]) {
                $matched = 1;
                $curbyte++;
            }else{
                $matched = 0;
                last;
            }
        }
        if($matched) {
            return $from + $startat;
        }else{
            $startat++;
        }
    }
    return undef;
}
sub _kCtrlS {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    $self->_Save();
    $self->{Busy} = 0;
    $self->_Stats();
}
sub _kCtrlT {
    my($self) = @_;
    $self->{Busy} = 1;
    $self->_Stats();    
    my $fs = new Hex::FileSelect();
    $fs->Title("Select which table file to use.");
    $fs->Start($Hex::TBL_DIR);
    my $path = $fs->Run();
    if($path ne '') {
        my $error = $self->{Table}->LoadFile($path);
        if($error) {
            my $msg = new Hex::MessageBox(
             "Error reading table file!",
             "$error", qw(OK));
            $msg->Run();
        }else{
            $self->{TblFilename} = (File::Spec->splitpath($path))[2];
            $self->{TblPath} = $path;
            $self->_Refresh();
            $self->_Highlight();
        }
    }
    $self->{Busy} = 0;
    $self->_Stats();
}

sub _AskToSave {
    my($self) = @_;
    if($self->{Modified}) {
        my $msg = new Hex::MessageBox(
         "Save Changes?",
         "The file has been modified since it was opened.\n".
         "Do you wish to save changes?",
         qw(Yes No));
        my $resp = $msg->Run();
        $self->_Save() if($resp == 0);
    }else{
    }
}

sub _Refresh {
    my($self) = @_;
    $self->{Output}->FillChar(' ', 80 * 46, 0, 2);
    my $buffer = substr($self->{Data}, $self->{Offset}, $self->{BPP});
    my @bytes = unpack('H*', $buffer);
    @bytes = $bytes[0] =~ /../g;
    my $line = 2;
   #iterates through each line
    while(my @line = splice(@bytes, 0, 16)) {
        $self->{Output}->Cursor(40, $line, 0, 0);
        my $sp = 1;
        my $linebuffer;
       #iterates through each byte in line
        foreach my $byte (@line) {
            $byte = uc $byte;
            $linebuffer .= $byte . (' ' x ($sp ^= 1));
            if(defined(my $c = $self->{Table}->Hex2Char($byte))) {
                if(defined $self->{Colors}->{length $c}) {
                    $self->{Output}->Attr($self->{Colors}->{length $c});
                }else{
                    $self->{Output}->Attr($main::FG_WHITE | $main::BG_BLACK);
                }
                $self->{Output}->Write($c);
            }else{
                $self->{Output}->Attr($main::FG_GRAY | $main::BG_BLACK);
                $self->{Output}->Write('.');
            }
        }
        $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_LIGHTCYAN, 39, 0, $line);
        $self->{Output}->WriteChar($linebuffer, 0, $line);
        last if(++$line > 47);
    }
    $self->{Output}->Cursor(0,0,0,0);
}

sub _Goto {
    my($self, $location) = @_;
    my $offset = $location & (~0^15); # ah... binary math..
    my $lastoffset = $self->_GetLastOffset();
    $offset = $lastoffset if($offset > $lastoffset);
    $location = $self->{Filesize} - 1 if($location >= $self->{Filesize});
    $self->{Offset} = $offset;
    $self->{Cursor} = $location;
    $self->_Refresh();
    $self->_Highlight();
    $self->_Stats();
}

sub _GotoFavorite {
    my($self, $number) = @_;
    if($self->{Favorites}->[$number] ne '') {
        $self->_Goto($self->{Favorites}->[$number]);
    }
}

sub _StoreFavorite {
    my($self, $number) = @_;
    $self->{Favorites}->[$number] = $self->{Cursor};
}

sub _OneLine {
    my($self, $offset, $row) = @_;
    my @bytes = unpack('H*', substr($self->{Data}, $offset, 16));
    @bytes = $bytes[0] =~ /../g;
    $self->{Output}->Cursor(40, $row, 0, 0);
    my $sp = 1;
    my $line;
    foreach my $byte (@bytes) {
        $byte = uc $byte;
        $line .= $byte . (' ' x ($sp ^= 1));
        if(defined(my $c = $self->{Table}->Hex2Char($byte))) {
            if(defined $self->{Colors}->{length $c}) {
                $self->{Output}->Attr($self->{Colors}->{length $c});
            }else{
                $self->{Output}->Attr($main::FG_WHITE | $main::BG_BLACK);
            }
            $self->{Output}->Write($c);
        }else{
            $self->{Output}->Attr($main::FG_GRAY | $main::BG_BLACK);
            $self->{Output}->Write('.');
        }
    }
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_LIGHTCYAN, 39, 0, $row);
    $self->{Output}->WriteChar($line, 0, $row);
}

sub _ScrollUp {
    my($self) = @_;
    if($self->{Offset} - 16 < 0) {
        #DO SQUAT
        return;
    }
    $self->{Output}->Scroll(
    0, 2, 79, 46, #left, top, right, bottom
    0, 3,   #to 0, 3
    ' ', $main::BG_BLACK | $main::FG_GRAY,
    0,0,79,49);

    $self->{Offset} -= 16;
    $self->_OneLine($self->{Offset}, 2);
    $self->_Stats();
}

sub _ScrollDown {
    my($self) = @_;
    if($self->{Offset} + $self->{BPP} >= length($self->{Data})) {
        #DO SQUAT
        return;
    }
    $self->{Output}->Scroll(
    0, 3, 79, 47, #left, top, right, bottom
    0, 2,   #to 0, 3
    ' ', $main::BG_BLACK | $main::FG_GRAY,
    0,0,79,49);
    $self->_OneLine($self->{Offset} + $self->{BPP}, 47);
    $self->{Offset} += 16;
    $self->_Stats();
}

sub _Highlight {
    my($self) = @_;
####Restore Last Highlighted Point
    $self->_RmvHL();
####Highlight new point
    my $pageoffset = $self->{Cursor} - $self->{Offset};
   #calc rows/cols for hex side
    my($curcol, $row) = $self->_GetCursorPos();
    my $col = ($pageoffset % 16) * 2 + ($pageoffset % 16) / 2;
   #calc rows/cols for text side
    my $amount =
     length($self->{Table}->Hex2Char( unpack('H*', substr( $self->{Data}, $self->{Cursor}, 1 ))));
   #for restoring on next highlight
    $self->{CursorAttrBuf} = $self->{Output}->ReadAttr($amount, $curcol, $row);
    $self->{LastCursor} = $self->{Cursor};
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_WHITE, 2, $col, $row);
    $self->{Output}->FillAttr($main::BG_RED | $main::FG_WHITE, $amount, $curcol, $row);
}

sub _RmvHL {
    my($self) = @_;
    my $pageoffset = $self->{LastCursor} - $self->{Offset};
   #Give up unless it's actually on the screen
    return unless($pageoffset >= 0 && $pageoffset < $self->{BPP});
   #calc rows/cols for hex side
    my($curcol, $row) = $self->_GetCursorPos($self->{LastCursor});
    my $col = ($pageoffset % 16) * 2 + ($pageoffset % 16) / 2;
   #calc rows/cols for text side
    my $amount =
     length($self->{Table}->Hex2Char( unpack('H*', substr( $self->{Data}, $self->{LastCursor}, 1 ))));

    my $color;
    if(defined $self->{Colors}->{$amount}) {
        $color = $self->{Colors}->{$amount};
    }else{
        $color = $main::FG_WHITE | $main::BG_BLACK;
    }
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_LIGHTCYAN, 2, $col, $row);
    $self->{Output}->FillAttr($color, $amount, $curcol, $row);

}

sub _GetCursorPos { #returns column/row numbers of first highlighted cell in char area
                    #based only on ->{Cursor} and ->{Offset}  w00t
    my($self, $cursor) = @_; #you can pass in a different offset for cursor to avoid standard
    $cursor = $self->{Cursor} unless(defined $cursor);
    my $pageoffset = $cursor - $self->{Offset};
    my $row = int($pageoffset / 16) + 2;
    my $prestring = substr(
                    $self->{Data},
                    $cursor - ($pageoffset % 16),
                    $pageoffset % 16);

    my $col = $self->{Table}->HexStr2Len(unpack('H*', $prestring), 1) + 40;
    return($col, $row); #x, y
}

sub _GetLastOffset { #returns last possible offset (one page from end)
    my($self) = @_;
    my $oddbytes = $self->{Filesize} % 16;
    my $adjust;
    $oddbytes ? ($adjust = 16) : ($adjust = 0);
    my $offset = $self->{Filesize} - $oddbytes + $adjust - $self->{BPP};
    if($offset >= 0) {
        return $offset
    }else{
        return 0;
    }


#     if($self->{Offset} + $self->{BPP} <= $self->{Filesize} - 1) {
#         if($self->{Offset} + 2 * $self->{BPP} > $self->{Filesize}) { #special adjust
#             $self->{Offset} = $self->{Filesize} - $oddbytes + $adjust - $self->{BPP};
#         }else{ #room for full page
#             $self->{Offset} += $self->{BPP};
#         }
#     }else{
#         #Do nothing.
#     }
}
                                   # 0000 0000 0000 0000
sub _Stats {
    my($self) = @_;
    my $editingline;
    if($self->{Editing}) {
        $editingline = "(Editing)";
        $self->{SelLength} = abs($self->{Cursor} - $self->{SelStart}) + 1;
    }

    my $selection = $self->{SelLength};
    my $SB1 = sprintf("Offset: %u (%Xh)", $self->{Cursor}, $self->{Cursor});
    $SB1 = $SB1 . sprintf("%*s", 80 - length($SB1), $editingline);

    my $SB2 = sprintf("Selection: %u (%Xh)", $selection, $selection);
    $SB2 .= sprintf("%*s", 80 - length($SB2), "Filesize: $self->{Filesize}");

    my $title = $self->{Busy} ? "JinX (Busy, please wait...)" : "JinX" . ' ' x 23;

    $self->_Status(
     TitleBar => $title . sprintf("%53s", "$self->{Filename}"
                                  . ($self->{Modified} ? '*' : '')
                                  ." ($self->{TblFilename})"),
     SB1 => $SB1,
     SB2 => $SB2,
    );
}

sub _Status {
    my($self, %args) = @_;

   #TitleBar Changes
    if(defined $args{TitleBar}) {
        $self->{Output}->WriteChar($args{TitleBar}, 0, 0);
    }
   #SB1 Changes
    if(defined $args{SB1}) {
        $self->{Output}->WriteChar($args{SB1}, 0, 48);
    }
   #SB2 Changes
    if(defined $args{SB2}) {
        $self->{Output}->WriteChar($args{SB2}, 0, 49);
    }

}

sub _Load {
    my($self, $path) = @_;
    if(open(FILE, $path)) {
        binmode(FILE);
        local $/ = undef;
        $self->{Data} = <FILE>;
        close FILE;
        $self->{Filename} = (File::Spec->splitpath($path))[2];
        $self->{Filesize} = length($self->{Data});
        $self->{Path} = $path;
    }else{
        my $msg = new Hex::MessageBox(
         "Failed to open file!",
         "$path could not be opened for the following reason:\n".
         "$!",
         qw(OK));
         $msg->Run();
    }
}

sub _Save {
    my($self) = @_;
    if($Hex::BACKUP) {
        copy($self->{Path},
             File::Spec->catfile($Hex::BACKUP_DIR, "$self->{Filename}.bak"));
    }

    if(open(FILE, ">$self->{Path}")) {
        binmode(FILE);
        print FILE $self->{Data};
        close FILE;
        $self->{Modified} = 0;
        $self->_Stats();
    }else{
        my $msg = new Hex::MessageBox(
         "Failed to open file!",
         "$self->{Path} could not be opened for the following reason:\n".
         "$!",
         qw(OK));
        $msg->Run();
    }

}

sub _SaveFavorites {
    my($self) = @_;
    if(open(FILE, ">$Hex::FAV_FILE")) {
        print FILE join(',', @{ $self->{Favorites} });
        close FILE;
    }else{
        my $msg = new Hex::MessageBox(
         "Failed to open file!",
         "Favorites settings could not be saved for this reason:\n".
         "$!",
         qw(OK));
        $msg->Run();
    }
}

sub _LoadFavorites {
    my($self) = @_;
    if(open(FILE, "$Hex::FAV_FILE")) {
        my $onlyline = <FILE>;
        chomp $onlyline;
        @{ $self->{Favorites} } = split(/,/, $onlyline);
        close FILE;
    }else{
        my $msg = new Hex::MessageBox(
         "Failed to open file!",
         "Favorites settings could not be loaded for this reason:\n".
         "$!",
         qw(OK));
        $msg->Run();
    }
}

sub _LoadFirstTableFile {
    my($self) = @_;
    my $path;
    if(-e File::Spec->catfile($Hex::TBL_DIR, '_ASCII.TBL')) {
        $path = File::Spec->catfile($Hex::TBL_DIR, '_ASCII.TBL');
    }else{
        my $fs = new Hex::FileSelect;
        $fs->Title("Select a table file");
        $fs->Start($Hex::TBL_DIR);
        $path = $fs->Run();
        undef $fs;
    }

    $self->{TblFilename} = (File::Spec->splitpath($path))[2];
    $self->{TblPath} = $path;
    if($path ne '') {
        my $error = $self->{Table}->LoadFile($path);
        die "$error\n" if($error);
    }
}

sub _Draw {                   
    my($self) = @_;
    $self->{Output}->Size(80, 50);
    $self->{Output}->Window(1, 0, 0, 640, 480);
    $self->{Input}->Mode(ENABLE_PROCESSED_INPUT);
#    $self->{Output}->Mode(ENABLE_PROCESSED_OUTPUT | ENABLE_WRAP_AT_EOL_OUTPUT);
    $self->{Output}->Cursor(-1, -1, 0, 0);
    $self->{Output}->Title("Hex");

#    $self->{Output}->FillChar('.', 80 * 50, 0, 0);
    $self->{Output}->FillAttr($main::BG_BLACK | $main::FG_GRAY, 80 * 50, 0, 0);
   #titlebar
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_WHITE, 80, 0, 0);
   #footer 
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_WHITE, 80, 0, 48);
    $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_WHITE, 80, 0, 49);    
    foreach(2..47) {
        $self->{Output}->FillAttr($main::BG_BLUE | $main::FG_LIGHTCYAN, 39, 0, $_);
    }

    $self->{Output}->Display();
}

sub _EndProg {
    my($self) = @_;
    $self->_SaveFavorites();
    exit;
}

1;
